/////////////////////////////////////////////////////////////////
//
//	Transform Feedback Example
//
//  Renders two rotating red triangles, and two green triangles that were output from the geometry shader
//  and were stored in the transform feedback buffer.
//  Also, it prints the contents of the transform feedback buffer every frame.
//
//	Derived from http://cvit.iiit.ac.in/
//
//  Gabriel Zachmann, 2011
/////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>

#ifdef _WIN32
#include <GL/glee.h>
#else
#define GL_GLEXT_PROTOTYPES
#endif

#if defined(__APPLE__) || defined(MACOSX)
#include <GLUT/glut.h>
#include <OpenGL/glext.h>
#else
#include <GL/glut.h>
#include <GL/glu.h>
#include <GL/gl.h>
#endif



GLuint tfvbo, vs, gs, fs, pg;
GLuint query, primitives_written = 0;

float t = 0.0;


const char * vertex_shader_source[] =
{
	"#version 120\n"
	"void main()"
	"{"
	"	gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;"
	"	gl_FrontColor = gl_Color;"
	"}"
};

const char * geometry_shader_saource[] =
{
"#version 120\n"
"#extension GL_EXT_geometry_shader4 : enable\n"
"#extension GL_EXT_gpu_shader4 : enable\n"
"void main(void)"
"{"
"	for ( int i = 0; i < gl_VerticesIn; i ++ )"
"	{"
"		gl_FrontColor = gl_FrontColorIn[i];"
"		gl_Position = gl_PositionIn[i];"
"		EmitVertex();"
"	}"
"	EndPrimitive();"
"}"
};

const char * fragment_shader_source[] =
{
	"#version 120\n"
	"void main()"
	"{"
	"	gl_FragColor = gl_Color;"
	"}"
};



// Print out shader info (debugging!)
void printShaderInfoLog( const char * kind, GLuint obj )
{
	int infologLength = 0;
	glGetShaderiv( obj, GL_INFO_LOG_LENGTH, &infologLength );

	if ( infologLength > 0 )
	{
		char * infoLog = (char *) malloc(infologLength);
		int charsWritten  = 0;
		glGetShaderInfoLog( obj, infologLength, &charsWritten, infoLog );
		printf("%s shader log: %s\n", kind, infoLog );
		free(infoLog);
	}
	else
	{
		printf("%s shader log: OK\n", kind );
	}
}


// Print out shader program info (debugging!)
void printProgramInfoLog( GLuint obj )
{
	int infologLength = 0;
	glGetProgramiv( obj, GL_INFO_LOG_LENGTH, &infologLength );
	
	if ( infologLength > 0 )
	{
		char * infoLog = (char *) malloc(infologLength);
		int charsWritten  = 0;
		glGetProgramInfoLog( obj, infologLength, &charsWritten, infoLog );
		printf("Program Log: %s\n",infoLog);
		free(infoLog);
	}
	else
	{
	 	printf("Program Log: OK\n");
	}
}


void checkGlErrors( void )
{
	GLenum e = glGetError();
	while ( e != GL_NO_ERROR )
	{
		fprintf( stderr, "GL error: %s!\n", gluErrorString(e) );
		e = glGetError();
	}
}



void setupTransformFeedbackBuffer(void)
{
	const char* attr[1] = {"gl_Position"};
	
	checkGlErrors();

	// generating the buffer, note that GL_TRANSFORM_FEEDBACK_BUFFER is NOT a buffer type
	glGenBuffers( 1, &tfvbo );
	glBindBuffer( GL_ARRAY_BUFFER, tfvbo );
	glBufferData( GL_ARRAY_BUFFER, 2*3*4*sizeof(float), NULL, GL_DYNAMIC_DRAW );
	
	// bind the TFB to get the feedback;  MUST be done here, not in display() !
	glBindBufferBase( GL_TRANSFORM_FEEDBACK_BUFFER_EXT, 0, tfvbo );
	
	glTransformFeedbackVaryingsEXT( pg, 1, attr, GL_INTERLEAVED_ATTRIBS_EXT );

	checkGlErrors();
}


void init(void)
{
	glClearColor( 0.5,0.5,0.5, 1 );
	
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	glOrtho( -1,1, -1,1, 0.2,2 );
	glMatrixMode( GL_MODELVIEW );
	
	vs = glCreateShader( GL_VERTEX_SHADER );
	gs = glCreateShader( GL_GEOMETRY_SHADER_EXT );
	fs = glCreateShader( GL_FRAGMENT_SHADER );
	glShaderSource( vs, 1, vertex_shader_source, NULL );
	glShaderSource( gs, 1, geometry_shader_saource, NULL );
	glShaderSource( fs, 1, fragment_shader_source, NULL );
	glCompileShader( vs );
	glCompileShader( gs );
	glCompileShader( fs );
	
	pg = glCreateProgram();
	glAttachShader( pg, vs );
	glAttachShader( pg, gs );
	glAttachShader( pg, fs );
	
	glProgramParameteriEXT( pg, GL_GEOMETRY_INPUT_TYPE_EXT, GL_TRIANGLES );				// must be done *before* linking the program!
	glProgramParameteriEXT( pg, GL_GEOMETRY_OUTPUT_TYPE_EXT, GL_TRIANGLE_STRIP );		// starting with OpenGL 3.2, these are set in the geom shader via the 'layout' syntax
	glProgramParameteriEXT( pg, GL_GEOMETRY_VERTICES_OUT_EXT, 3 );
	
	setupTransformFeedbackBuffer();														//set up transform feedback buffer

	glLinkProgram( pg );
	glUseProgram( pg );
	
	printShaderInfoLog( "Vertex", vs );
	printShaderInfoLog( "Geometry", gs );
	printShaderInfoLog( "Fragment", fs );
	printProgramInfoLog( pg );

	glGenQueries(1, &query);
	
	checkGlErrors();
}


void display(void)
{
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	glLoadIdentity();
	t += 0.001;
	glRotatef( t, 0.0, 0.0, 1.0 );
	
	// ------ display the normal rendered object at this viewport, with color red
	glColor3f( 1,0,0 );
	glViewport( 0,0, 256,256 );
	
	// start transform feedback so that vertices get targetted to 'tfvbo'
	glBeginTransformFeedbackEXT( GL_TRIANGLES );
	glBeginQuery( GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_EXT, query );

	// render primitives and copy these positions to tranform feedback buffer
	glBegin( GL_TRIANGLES );
	glVertex4f( -0.5,  -0.5,  -0.3, 1 );
	glVertex4f( -0.6,  -0.45, -0.3, 1 );
	glVertex4f( -0.45, -0.47, -0.3, 1 );

	glVertex4f( 0.5,  0.5,  -0.3, 1 );
	glVertex4f( 0.6,  0.45, -0.3, 1 );
	glVertex4f( 0.45, 0.47, -0.3, 1 );
	glEnd();
	glEndQuery( GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN_EXT );

	// end transform feedback
	glEndTransformFeedbackEXT();
	
	// read back query results
	glGetQueryObjectuiv( query, GL_QUERY_RESULT, &primitives_written );
	if ( primitives_written != 2 )
		fprintf( stderr, "Primitives written to TFB: %d !\n", primitives_written );
	
	// retrieve the data stored in the TFB
	checkGlErrors();
	glBindBuffer( GL_ARRAY_BUFFER, tfvbo );
	float * TFBdata = static_cast<float*>( glMapBuffer(GL_ARRAY_BUFFER, GL_READ_ONLY) );
	if ( TFBdata == NULL )
		checkGlErrors();
	else
	{
		fputs("TFB contents: ", stdout);
		for ( int i = 0; i < 2*3*4; i ++ )
			printf( "% 10f  ", TFBdata[i] );
		putchar('\n');
	}
	bool success = glUnmapBuffer( GL_ARRAY_BUFFER );
	if ( ! success )
		checkGlErrors();																// in this case, tfvbo can't be used any more!
	glBindBuffer( GL_ARRAY_BUFFER, 0 );													// not absolutely necessary

	// ------ display the transform feedback recorded data to this viewport
	glColor3f( 0,1,0 );
	glViewport( 256,0, 256,256 );
	
	// render the buffer which has recieved data from transform feedback;
	// note that glRotate is still active, and it has already rotated once.
	// Consequently, the green triangles will rotate twice as fast.
	glEnableClientState( GL_VERTEX_ARRAY );
	glBindBuffer( GL_ARRAY_BUFFER, tfvbo );
	glVertexPointer( 4, GL_FLOAT, 0, 0 );												// '4' because we have used glVertex4f() above
	glDrawArrays( GL_TRIANGLES, 0, 2*3 );
	glDisableClientState( GL_VERTEX_ARRAY );
	
	glutSwapBuffers();
}


int main(int argc, char **argv)
{
	glutInit( &argc, argv );
	glutInitDisplayMode( GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH );
	glutInitWindowSize( 512,256 );
	glutCreateWindow( "Transform Feedback Demo" );
	glutDisplayFunc( display );
	glutIdleFunc( display );
	init();
	glutMainLoop();
	return 0;
}
